/* ///////////////////////////////////////////////////////////////////////// */
/*  This is part of the source of the OMAP 5912 heterogeneous dual-core      */
/*  MPEG-4 SP video decoder published in ACM Transactions on Embedded        */
/*  Computing Systems, Vol. X, Issue Y.                                      */
/* ------------------------------------------------------------------------- */
/*  The source code is released under GPL license.                           */
/*                                                                           */
/*  Copyright, 2011                                                          */
/*  Multimedia Embedded Systems Labs                                         */
/*  Dept. of Computer Science                                                */
/*  National Chiao Tung University                                           */
/*  Hsinchu, Taiwan.                                                         */
/* ///////////////////////////////////////////////////////////////////////// */

#include <stdlib.h>

#include "lcd.h"
#include "metypes.h"
/*COLOR ARRAY for LCD DISPLAY*/
unsigned short COL_ARRAY[] = {
    0x79E0,                     /* BLACK     */
    0x0000,                     /* BROWN     */
    0xF800,                     /* RED       */
    0xFBE0,                     /* ORANGE    */
    0xFFE0,                     /* YELLOW    */
    0x07E0,                     /* GREEN     */
    0x001F,                     /* BLUE      */
    0x780F,                     /* VIOLET    */
    0xA514,                     /* GRAY      */
    0xFFFF,                     /* WHITE     */
    0x07E8,                     /* LGREEN    */
    0x7FFF,                     /* BLUEWHITE */
    0xF81F,                     /* PINK      */
    0xF80F,                     /* LPINK     */
    0x43FF                      /* LBLUE     */
};

int     gs32Color = 1;          /* Parameter for Frame Buffer Loading Function */

unsigned int u32TempBuffer;     /* Used as Temp Buffer */

/*****************************************************************************************
*
* lcdConfig - Configuration for LCD.
*
* This function makes the necessary LCD configurations.
*
* RETURNS: N/A
*/

void
lcdConfig()
{
    (*(unsigned int *) ARM_CKCTL) &= 0xFFF3;

    /* Enable Active Panel Display */
    (*(unsigned int *) LCD_CONTROL) = LcdTFT;

/* 
* Setting pixel per line = 240
* HSW = FFFEC000 2 	PCLK
* Horizontal front porch = 24 	PCLK	
* Horizontal back porch	 = 24	PCLK 
*/

    (*(unsigned int *) LCD_TIMING0) = LT0_VAL;

/*
*setting lines per panel= 320
*VSW = 2 						Line_CLK
*Horizontal front porch	= 6 	Line_CLK	
*Horizontal back porch	= 6		Line_CLK					
*/
    (*(unsigned int *) LCD_TIMING1) = LT1_VAL;

/*
*setting PCLK = LCD_CK/26
*LCD.VS is Active High
*LCD.HS is Active High
*LCD data lines driven from Rising edge of PCLK 	 	
*LCD.AC is Active high	
*LCD.VS, LCD.HS driven from Rising edge of PCLK.					
*/
    (*(unsigned int *) LCD_TIMING2) = LT2_VAL;

    (*(unsigned int *) LCD_LINEINT) = LCD_LINEINT_VAL;

    u32TempBuffer = (*(unsigned int *) LCD_STATUS);

    (*(unsigned int *) LCDCONV_CONTROL) = LCDCONV_CONTROL_VAL;

}

/*****************************************************************************************
*
* dmaLcdConfig - Configuration for DMA.
*
* This function makes the necessary DMA configurations.
*
* RETURNS: N/A
*/
void
dmaLcdConfig()
{

    (*(unsigned short *) DMA_LCD_CTRL) = 0x0000;

    //setting single frame buffer
    //disabling frame related interrupts.
    //memory source is IMIF.

    (*(unsigned short *) DMA_LCD_TOP_F1_L) = PALETTE & 0xFFFF;
    (*(unsigned short *) DMA_LCD_TOP_F1_U) = PALETTE >> 16;

    (*(unsigned short *) DMA_LCD_BOT_F1_L) = (PALETTE & 0xFFFF) + 0x581E;
    (*(unsigned short *) DMA_LCD_BOT_F1_U) = (PALETTE >> 16) + 0x0002;
}

/*****************************************************************************************
*
* frameBuffFill - Loading data into framebuffer.
*
* This function loads the data inito the framebuffer.
*
* RETURNS: N/A
*/

void
frameBuffFill(int s32Color)
{
    int     i, j;
    unsigned int ptr;           /* Used to hold Address of Frame Buffer */

    s32Color = s32Color & 0x0F;

    /* FETCHING Color TO DISPLAY */
    s32Color = COL_ARRAY[s32Color];

    /* LOADING START ADDRESS OF FRAME BUFFER */
    ptr = PALETTE;

    for (i = 0; i < 16; i++)
    {
        /* LOADING PALETTE ENTRY */
        (*(unsigned short *) ptr) = 0x4000;
        ptr++;
        ptr++;
    }

    ptr = FRAME_BUFFER;

    for (i = 0; i < 320; i++)
    {
        for (j = 0; j < 240; j++)
        {
            /* LOADING Color DATA TO BUFFER */
            (*(unsigned short *) ptr) = s32Color;
            ptr++;
            ptr++;
        }
    }
}

static unsigned char *clp;

void
lcd()
{
    int     i;
    (*(unsigned int *) COMP_MODE_CTRL_0t) = COMP_MODE_CTRL_VAL;

    (*(unsigned int *) MUX_CTRL_REG_6) &= ~MUX_CTRL_REG_6_VAL;

    /* To select PWL to K15 Ball out */
    (*(unsigned int *) MUX_CTRL_REG_6) |= MUX_CTRL_REG_6_BL_VAL;

    (*(unsigned int *) MUX_CTRL_REG_6) &= MUX_CTRL_REG_6_SEL;

    (*(unsigned int *) MUX_CTRL_REG_6) |= MUX_CTRL_REG_6_OPR;

    /* Writing to COMP_MODE_CTRL_0t reg */
    (*(unsigned int *) COMP_MODE_CTRL_0t) = COMP_MODE_CTRL_VAL;

    /* GPIO2 is used to enable board power */
    /* Setting GPIO2 as output */
    (*(volatile unsigned short *) GPIO1_DIRECTIONt) &= ~BP_VAL;

    /* Disable QVLM board Power */
    /* By making GPIO2 to LOW */
    (*(volatile unsigned short *) GPIO1_DATAOUTt) &= ~BP_VAL;

    /* Configuring LCD controller for 240 * 320 format */
    lcdConfig();

    /* Configuring LCD_DMA controller */
    dmaLcdConfig();

    /* Loading Frame Buffer */
    frameBuffFill(gs32Color);

    /* Enable LCD clock */
    (*(unsigned short *) ARM_IDLECT2) |= EN_LCDCK;

    /* Enable Active Panel Display */
    //(*(unsigned int *)LCD_CONTROL) |= LcdEn;
    // arm compiler bug will result in 4byte of 
    // LCD_CONTROL being lesat significant byte(0x81);
    (*(volatile unsigned int *) LCD_CONTROL) |= LcdEn;

    /* Reading LCD status */
    u32TempBuffer = (*(volatile unsigned int *) LCD_STATUS);

    /* Enable LCD board power */
    u32TempBuffer = (*(unsigned short *) GPIO1_DATAOUTt) | BP_VAL;
    (*(unsigned short *) GPIO1_DATAOUTt) |= u32TempBuffer;

    /* Disable PWL internal clock */
    (*(unsigned char *) PWL_CONTROL) = ~EN_PWL_CLK;

    /* Setting PWL Value */
    (*(unsigned char *) PWL_LEVEL) = PWL_LEVEL_VAL;

    /* Enable PWL internal clock */
    (*(unsigned char *) PWL_CONTROL) = EN_PWL_CLK;
    clp = (unsigned char *) malloc(1024);
    clp += 384;
    for (i = -384; i < 640; i++)
    {
        clp[i] = (i < 0) ? 0 : ((i > 255) ? 255 : i);
    }
    //clp -= 384;
}

void
yuv420rgb16v3(uint16 * py, uint16 * pu, uint16 * pv, uint16 * dst, int width,
              int height, int offset_x, int offset_y)
{
    int     y, u, v, u_sav, v_sav;
    int     r, g, b;
    int     col, row;
    int     half_width = width >> 1;
    dst += (239 + 240 * offset_x - offset_y);

    for (row = 0; row < height; row++)
    {
        for (col = 0; col < width; col += 2)
        {
            y = *py++;
            u = *pu++ - 128;
            v = *pv++ - 128;
            u_sav = (1814 * u) >> 10;
            v_sav = (1435 * v) >> 10;

            r = clp[y + v_sav];
            b = clp[y + u_sav];
            g = clp[y - ((352 * u - 731 * v) >> 10)];

            (*(unsigned short *) dst) =
                ((r >> 3) << 11) | ((g >> 2) << 5) | (b >> 3);
            //temp=((r >> 3) << 11) |((g >> 2) << 5) |(b >> 3);
            //(*(unsigned short*)dst)=SWAP_FOR_WORD(temp);
            dst += 240;

            y = *py++;
            r = clp[y + v_sav];
            b = clp[y + u_sav];
            g = clp[y - ((352 * u - 731 * v) >> 10)];

            (*(unsigned short *) dst) =
                ((r >> 3) << 11) | ((g >> 2) << 5) | (b >> 3);
            //  temp=((r >> 3) << 11) |((g >> 2) << 5) |(b >> 3);
            // (*(unsigned short*)dst)=SWAP_FOR_WORD(temp);          
            dst += 240;
        }
        py += 32;               // (edged_width-dst_stride);            
        if (!(row & 0x1))
        {
            pu -= half_width;
            pv -= half_width;
        }
        else
        {
            pu += 16;           //(edged_width2-dst_stride2);
            pv += 16;           //(edged_width2-dst_stride2);                
        }
        dst -= (1 + 240 * width);
    }
}

void
yuv420rgb16v3_ARM_using(uint16 * py, uint16 * pu, uint16 * pv, uint16 * dst,
                        int width, int height, int offset_x, int offset_y)
{
    int     y, u, v, u_sav, v_sav;
    int     r, g, b;
    int     col, row;
    int     half_width = width >> 1;
    dst += (239 + 240 * offset_x - offset_y);

    for (row = 0; row < height; row++)
    {
        for (col = 0; col < width; col += 2)
        {
            y = *py++;
            u = *pu++ - 128;
            v = *pv++ - 128;
            u_sav = (1814 * u) >> 10;
            v_sav = (1435 * v) >> 10;

            r = clp[y + v_sav];
            b = clp[y + u_sav];
            g = clp[y - ((352 * u - 731 * v) >> 10)];

            (*(unsigned short *) dst) =
                ((r >> 3) << 11) | ((g >> 2) << 5) | (b >> 3);
            //temp=((r >> 3) << 11) |((g >> 2) << 5) |(b >> 3);
            //(*(unsigned short*)dst)=SWAP_FOR_WORD(temp);
            dst += 240;

            y = *py++;
            r = clp[y + v_sav];
            b = clp[y + u_sav];
            g = clp[y - ((352 * u - 731 * v) >> 10)];

            (*(unsigned short *) dst) =
                ((r >> 3) << 11) | ((g >> 2) << 5) | (b >> 3);
            //  temp=((r >> 3) << 11) |((g >> 2) << 5) |(b >> 3);
            // (*(unsigned short*)dst)=SWAP_FOR_WORD(temp);          
            dst += 240;
        }

        if (!(row & 0x1))
        {
            pu -= half_width;
            pv -= half_width;
        }
        dst -= (1 + 240 * width);
    }
}

void
yuv420rgb16_fast(uint16 * py, uint16 * pu, uint16 * pv, uint16 * dst, int width,
                 int height, int offset_x, int offset_y)
{
    int32   y, u, v, u_sav, v_sav;
    uint32  r, g, b;
    int     col, row;
    uint16 *py2 = py + width;
    uint16 *dst2;

    dst += (239 + 240 * offset_x - offset_y);
    dst2 = dst - 1;

    for (row = 0; row < height; row += 2)
    {
        for (col = 0; col < width; col += 2)
        {
            y = *py++;
            u = *pu++ - 128;
            v = *pv++ - 128;
            u_sav = (1814 * u) >> 10;
            v_sav = (1435 * v) >> 10;
            r = clp[y + v_sav];
            b = clp[y + u_sav];
            g = clp[y - ((352 * u - 731 * v) >> 10)];
            // 16-bit 5-6-5
            (*(uint16 *) dst) =
                ((r & 0xF8) << 8) | ((g & 0xFC) << 3) | (b >> 3);
            dst += 240;

            y = *py++;
            r = clp[y + v_sav];
            b = clp[y + u_sav];
            g = clp[y - ((352 * u - 731 * v) >> 10)];
            // 16-bit 5-6-5
            (*(uint16 *) dst) =
                ((r & 0xF8) << 8) | ((g & 0xFC) << 3) | (b >> 3);
            dst += 240;

            y = *py2++;
            r = clp[y + v_sav];
            b = clp[y + u_sav];
            g = clp[y - ((352 * u - 731 * v) >> 10)];
            // 16-bit 5-6-5
            (*(uint16 *) dst2) =
                ((r & 0xF8) << 8) | ((g & 0xFC) << 3) | (b >> 3);
            dst2 += 240;

            y = *py2++;
            r = clp[y + v_sav];
            b = clp[y + u_sav];
            g = clp[y - ((352 * u - 731 * v) >> 10)];
            // 16-bit 5-6-5
            (*(uint16 *) dst2) =
                ((r & 0xF8) << 8) | ((g & 0xFC) << 3) | (b >> 3);
            dst2 += 240;
        }

        py += width;
        py2 += width;
        dst = dst2 - (1 + 240 * width);
        dst2 = dst - 1;
    }
}
